<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

/**
 * Class User
 * @package app\models
 * @property int $id
 * @property string $email
 * @property string $password_hash
 * @property string $auth_key
 * @property string $full_name
 * @property string $phone
 * @property string $passport
 * @property string $address
 * @property string $role
 * @property int $status
 * @property int $created_at
 * @property int $updated_at
 */
class User extends ActiveRecord implements IdentityInterface
{
    const ROLE_USER = 'user';
    const ROLE_ADMIN = 'admin';
    
    const STATUS_DELETED = 0;
    const STATUS_ACTIVE = 10;

    public $password;
    public $password_repeat;
    public $agree_terms;

    const SCENARIO_REGISTER = 'register';
    const SCENARIO_UPDATE = 'update';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%user}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            // Обязательные поля для регистрации
            [['email', 'password', 'password_repeat', 'full_name', 'phone', 'passport', 'address', 'agree_terms'], 'required', 'on' => self::SCENARIO_REGISTER],
            
            // Для обновления не требуем пароль
            [['email', 'full_name', 'phone', 'passport', 'address'], 'required', 'on' => self::SCENARIO_UPDATE],
            
            // Валидация email
            ['email', 'email'],
            ['email', 'unique', 'message' => 'Этот email уже зарегистрирован'],
            
            // Валидация пароля только при регистрации
            ['password', 'string', 'min' => 7, 'message' => 'Пароль должен содержать минимум 7 символов', 'on' => self::SCENARIO_REGISTER],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'message' => 'Пароли не совпадают', 'on' => self::SCENARIO_REGISTER],
            
            // Валидация телефона
            ['phone', 'match', 'pattern' => '/^\+7\(\d{3}\)\d{3}-\d{2}-\d{2}$/', 
             'message' => 'Телефон должен быть в формате +7(XXX)XXX-XX-XX'],
            
            // Валидация ФИО (только кириллица и пробелы)
            ['full_name', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s]+$/u', 
             'message' => 'ФИО должно содержать только русские буквы и пробелы'],
            
            // Проверка согласия с правилами только при регистрации
            ['agree_terms', 'compare', 'compareValue' => true, 
             'message' => 'Вы должны согласиться с правилами регистрации', 'on' => self::SCENARIO_REGISTER],
            
            // Дополнительные валидации
            [['full_name'], 'string', 'max' => 150],
            [['phone'], 'string', 'max' => 20],
            [['passport'], 'string', 'max' => 50],
            [['address'], 'string'],
            
            // Роль
            ['role', 'in', 'range' => [self::ROLE_USER, self::ROLE_ADMIN]],
            ['role', 'default', 'value' => self::ROLE_USER],
            
            // Статус
            ['status', 'default', 'value' => self::STATUS_ACTIVE],
            ['status', 'in', 'range' => [self::STATUS_ACTIVE, self::STATUS_DELETED]],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function scenarios()
    {
        $scenarios = parent::scenarios();
        $scenarios[self::SCENARIO_REGISTER] = ['email', 'password', 'password_repeat', 'full_name', 'phone', 'passport', 'address', 'agree_terms', 'role', 'status'];
        $scenarios[self::SCENARIO_UPDATE] = ['email', 'full_name', 'phone', 'passport', 'address', 'role', 'status'];
        return $scenarios;
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'email' => 'Email',
            'password' => 'Пароль',
            'password_repeat' => 'Повтор пароля',
            'full_name' => 'ФИО',
            'phone' => 'Телефон',
            'passport' => 'Серия и номер паспорта',
            'address' => 'Адрес проживания',
            'agree_terms' => 'Согласен с правилами регистрации',
            'role' => 'Роль',
            'status' => 'Статус',
            'created_at' => 'Дата регистрации',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                $this->auth_key = Yii::$app->security->generateRandomString();
                $this->password_hash = Yii::$app->security->generatePasswordHash($this->password);
                $this->created_at = time();
            }
            $this->updated_at = time();
            return true;
        }
        return false;
    }

    // ==================== IdentityInterface методы ====================

    /**
     * {@inheritdoc}
     */
    public static function findIdentity($id)
    {
        return static::findOne(['id' => $id, 'status' => self::STATUS_ACTIVE]);
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->getPrimaryKey();
    }

    /**
     * {@inheritdoc}
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }

    /**
     * {@inheritdoc}
     */
    public function validateAuthKey($authKey)
    {
        return $this->getAuthKey() === $authKey;
    }

    /**
     * Валидация пароля
     */
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    /**
     * Поиск пользователя по email
     */
    public static function findByEmail($email)
    {
        return static::findOne(['email' => $email, 'status' => self::STATUS_ACTIVE]);
    }

    // ==================== Дополнительные методы ====================

    /**
     * Проверка, является ли пользователь администратором
     */
    public function isAdmin()
    {
        return $this->role === self::ROLE_ADMIN;
    }

    /**
     * Получение всех ролей
     */
    public static function getRoles()
    {
        return [
            self::ROLE_USER => 'Пользователь',
            self::ROLE_ADMIN => 'Администратор',
        ];
    }

    /**
     * Получение всех статусов
     */
    public static function getStatuses()
    {
        return [
            self::STATUS_ACTIVE => 'Активен',
            self::STATUS_DELETED => 'Удален',
        ];
    }

    /**
     * Связь с заявками
     */
    public function getApplications()
    {
        return $this->hasMany(Application::class, ['user_id' => 'id']);
    }

    /**
     * Связь с отзывами
     */
    public function getReviews()
    {
        return $this->hasMany(Review::class, ['user_id' => 'id']);
    }

    /**
     * Форматирование телефона для отображения
     */
    public function getFormattedPhone()
    {
        return $this->phone;
    }
}