<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use app\models\Tour;
use app\models\Review;

/**
 * SiteController - главный контроллер
 */
class SiteController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['logout'],
                'rules' => [
                    [
                        'actions' => ['logout'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function actions()
    {
        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
        ];
    }

    /**
     * Главная страница
     */
    public function actionIndex()
    {
        // Получаем 6 активных туров для главной
        $tours = Tour::find()
            ->where(['is_active' => 1])
            ->orderBy(['created_at' => SORT_DESC])
            ->limit(6)
            ->all();
        
        // Получаем последние отзывы
        $reviews = Review::find()
            ->where(['is_published' => 1])
            ->with(['user', 'tour'])
            ->orderBy(['created_at' => SORT_DESC])
            ->limit(3)
            ->all();
            
        return $this->render('index', [
            'tours' => $tours,
            'reviews' => $reviews,
        ]);
    }

    /**
     * Каталог туров
     */
    public function actionTours()
    {
        $query = Tour::find()->where(['is_active' => 1]);
        
        // Фильтрация по цене
        $minPrice = Yii::$app->request->get('min_price');
        $maxPrice = Yii::$app->request->get('max_price');
        
        if ($minPrice !== null && $minPrice !== '') {
            $query->andWhere(['>=', 'price', $minPrice]);
        }
        
        if ($maxPrice !== null && $maxPrice !== '') {
            $query->andWhere(['<=', 'price', $maxPrice]);
        }
        
        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 9,
            ],
            'sort' => [
                'defaultOrder' => [
                    'created_at' => SORT_DESC,
                ]
            ],
        ]);

        return $this->render('tours', [
            'dataProvider' => $dataProvider,
            'minPrice' => $minPrice,
            'maxPrice' => $maxPrice,
        ]);
    }

    /**
     * Детальная страница тура
     */
    public function actionTour($id)
    {
        $tour = Tour::findOne($id);
        
        if (!$tour || !$tour->is_active) {
            throw new \yii\web\NotFoundHttpException('Тур не найден');
        }
        
        // Получаем отзывы к туру
        $reviews = Review::find()
            ->where(['tour_id' => $id, 'is_published' => 1])
            ->with('user')
            ->orderBy(['created_at' => SORT_DESC])
            ->all();
        
        // Средний рейтинг
        $avgRating = Review::find()
            ->where(['tour_id' => $id, 'is_published' => 1])
            ->average('rating') ?? 0;

        return $this->render('tour', [
            'tour' => $tour,
            'reviews' => $reviews,
            'avgRating' => round($avgRating, 1),
        ]);
    }

    /**
     * Страница "О нас"
     */
    public function actionAbout()
    {
        return $this->render('about');
    }

    /**
     * Страница контактов
     */
    public function actionContact()
    {
        return $this->render('contact');
    }
}