<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use app\models\User;
use app\models\LoginForm;

/**
 * AuthController - управление регистрацией и авторизацией
 */
class AuthController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'actions' => ['register', 'login'],
                        'allow' => true,
                        'roles' => ['?'], // Только для гостей
                    ],
                    [
                        'actions' => ['logout'],
                        'allow' => true,
                        'roles' => ['@'], // Только для авторизованных
                    ],
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function actions()
    {
        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
        ];
    }

    /**
     * Регистрация нового пользователя
     */
    public function actionRegister()
    {
        if (!Yii::$app->user->isGuest) {
            return $this->goHome();
        }

        $model = new User();
        $model->scenario = User::SCENARIO_REGISTER; // Устанавливаем сценарий

        if ($model->load(Yii::$app->request->post())) {
            if ($model->save()) {
                // Автоматическая авторизация после регистрации
                Yii::$app->user->login($model, 3600 * 24 * 30);
                
                Yii::$app->session->setFlash('success', 
                    '<i class="fas fa-check-circle"></i> Регистрация прошла успешно! Добро пожаловать, ' . $model->full_name . '!'
                );
                
                return $this->redirect(['site/index']);
            } else {
                Yii::$app->session->setFlash('error', 
                    '<i class="fas fa-exclamation-circle"></i> Пожалуйста, исправьте ошибки в форме.'
                );
            }
        }

        return $this->render('register', [
            'model' => $model,
        ]);
    }

    /**
     * Авторизация пользователя
     */
    public function actionLogin()
    {
        if (!Yii::$app->user->isGuest) {
            return $this->goHome();
        }

        $model = new LoginForm();
        
        if ($model->load(Yii::$app->request->post())) {
            if ($model->login()) {
                Yii::$app->session->setFlash('success', 
                    '<i class="fas fa-check-circle"></i> Вы успешно авторизовались!'
                );
                
                // Перенаправляем администратора в админку
                if (Yii::$app->user->identity->isAdmin()) {
                    return $this->redirect(['admin/index']);
                }
                
                return $this->goBack();
            } else {
                Yii::$app->session->setFlash('error', 
                    '<i class="fas fa-exclamation-circle"></i> Неверный email или пароль.'
                );
            }
        }

        // Очищаем пароль для безопасности
        $model->password = '';

        return $this->render('login', [
            'model' => $model,
        ]);
    }

    /**
     * Выход из системы
     */
    public function actionLogout()
    {
        Yii::$app->user->logout();
        
        Yii::$app->session->setFlash('success', 
            '<i class="fas fa-sign-out-alt"></i> Вы успешно вышли из системы.'
        );
        
        return $this->goHome();
    }
}